--[[

Developer PowerTools is a quality-of-life mod for modders that works well alongside Power Tools and/or Easy Dev Controls.

Author:     w33zl / WZL Modding (facebook.com/w33zl)
Version:    2.0.0
Modified:   2024-11-15

GitHub:     github.com/w33zl/FS25_DevTools

Changelog:
    2.0.0       FS25 Upgrade

]]

DevTools = Mod:init()

DevTools:trySource("lib/MemoryProfiler.lua")
DevTools:trySource("lib/StringWriter.lua")
DevTools:source("lib/DevHelper.lua")

local OUTPUT_HEADER = [[
-- This file was automatically generated by the mod FS25 Developer PowerTools (https://github.com/w33zl/FS25_DevTools)
]]

local l_currentModSettingsDirectory = g_currentModSettingsDirectory

local function getOrInitGlobalMod(name)
    g_globalMods[name] = g_globalMods[name] or {}
    return g_globalMods[name]
end

_G.g_powerTools = getOrInitGlobalMod("FS25_PowerTools")

local function safeAddConsoleCommand(name, description, ...)
    g_powerTools.registeredConsoleCommands = g_powerTools.registeredConsoleCommands or {}

    if g_powerTools.registeredConsoleCommands[name] then
        Log:debug("Console command '%s' already registered, skipping.", name)
        return
    end

    local args = { ... }
    local success = pcall(function()
        addConsoleCommand(name, description, unpack(args))
    end)

    if success then
        g_powerTools.registeredConsoleCommands[name] = true
    else
        Log:warning("Failed to register console command '%s'", name)
    end
end

local function quitGame(restart, hardReset)
    restart = restart or false
    hardReset = hardReset or false

    if hardReset then
        doRestart(true, "-cheats -skipStartVideos")
        return
    else
        SystemConsoleCommands.softRestart()
    end

    ---Restarts the application
-- Has to be called at the end of the frame because engine restartApplicaiton function queues the actual restart
-- and performs it at the end of an event (draw, update, mouseInput etc)


    -- Log:var("performRestart", restart)
    -- Log:var("restartApplication", restartApplication)
    -- Log:var("doRestart", doRestart)
    -- Log:table("RestartManager", RestartManager)
    -- Log:var("SystemConsoleCommands.softRestart", SystemConsoleCommands.softRestart)
    -- -- RestartManager

    -- restartApplication("", "")

    --doRestart?
    ---Queues a restart
-- @param boolean restartProcess
-- @param string args

    -- local success = pcall(function()
    --     if not hardReset and g_currentMission ~= nil then
    --         OnInGameMenuMenu()
    --     end

    --     RestartManager:setStartScreen(RestartManager.START_SCREEN_MAIN)

    --     local gameID = ""
    --     if restart and g_careerScreen ~= nil and g_careerScreen.currentSavegame ~= nil then
    --         gameID = g_careerScreen.currentSavegame.savegameIndex
    --     end

    --     -- doRestart(hardReset, "-autoStartSavegameId " .. gameID)
    --     doRestart(hardReset, "")

    --     --TODO: (optionally) clear the log?
    -- end)

    if not success then
        Log:error("Failed to exit/restart game")
    end
end

function DevTools:saveTable(tableName, actualTable, filePath, maxDepth, ignoredTables, memProfiler, header)
    collectgarbage("collect") -- Ensure we have maximum mem available for this memory consuming operation
    -- collectgarbage("setpause")
    DevHelper.saveTableToFile(filePath, tableName, actualTable, maxDepth, ignoredTables, nil, memProfiler, header)
    -- collectgarbage("restart")
    collectgarbage("collect") -- Also cleanup to not leave a mess
end

local DevToolsConsoleCommands = {
	softRestart = function (self)
		-- RestartManager:setStartScreen(RestartManager.START_SCREEN_MAIN)
		-- -- doRestart(false, "")
        -- g_gui:showGui("MainScreen")
        -- OnInGameMenuMenu()
        quitGame(true, false)
	end,
	hardReset = function (self)
		quitGame(true, true)
	end,

    testRestart = function (self)
        -- doRestart(false, "-cheats -skipStartVideos")
        -- restartApplication(false, "-cheats -skipStartVideos")

        -- doRestart(true, "-cheats -skipStartVideos")
        -- StartParams.setValue("autoStartSavegameId", 1)
        -- StartParams.setValue("restart", nil)


        g_pendingRestartData = g_pendingRestartData or {}
        g_pendingRestartData.restartProcess = false
        g_pendingRestartData.args = ""

        -- if g_pendingRestartData == nil then
        --     autoStartLocalSavegame(1)
        -- end        
        -- StartParams.setValue
        -- autoStartLocalSavegame(1)
        -- RestartManager:setStartScreen(v10)
        -- doRestart(false, "")
        -- if g_currentMission == nil then
        --     RestartManager:setStartScreen(RestartManager.START_SCREEN_MAIN)
        --     doRestart(false, "")
        -- else
        --     OnInGameMenuMenu()
        -- end
        -- GameLoadingCancelSimulator
        -- g_pendingRestartData.args = string.format("-gameLoadingCancelSimulator %d", v_u_2)
        -- performRestart
    end,

    extractTable = function (self, tableName)
        if tableName == nil then
            Log:error("tableName is mandatory")
            return nil
        end
        local actualTable = nil

        if tableName == "_G" then
            Log:debug("Getting global")
            actualTable = _G
        elseif tableName == "__G" then
            Log:debug("Getting true global")
            actualTable = DevTools.__g
        else
            -- Get table from global mod scope (sandbox)
            actualTable = _G[tableName]

            -- Get table from true global scope
            if actualTable == nil then
                actualTable = DevTools.__g[tableName]
            end

            -- Try to get table from mod environment
            if actualTable == nil then
                actualTable = DevTools.env[tableName]
            end

            -- Try to get table via loadstring/pcall method (supports multi level tables as A.B.C)
            if actualTable == nil then
                local __G = DevTools.__g
                local tableExtractFunction = loadstring("return " .. tableName)
                local success, tempTable = pcall(tableExtractFunction) -- First try with the regular environment

                if success then
                    actualTable = tempTable
                else
                    Log:debug("Trying fallback enviroment")
                    setfenv(tableExtractFunction, DevTools.__g)
                    success, tempTable = pcall(tableExtractFunction) -- First try with the regular environment
                end

                if success then
                    actualTable = tempTable
                else
                    Log:debug("Even loadstring/pcall method failed on table '%s'", tableName)
                end
            end

        end

        if actualTable == nil then
            Log:error("Table '%s' not found", tableName)
        end

        return actualTable
	end,

    printTable = function (self, tableName, maxDepth)
        local USAGE = "USAGE: # dtPrintTable tableName [maxDepth]"
        if tableName == nil then
            Log:error("tableName is mandatory")
            return USAGE
        end
        maxDepth = tonumber(maxDepth)
        maxDepth = maxDepth or 1

        if type(maxDepth) ~= "number" then
            Log:error("maxDepth must be a number")
            return USAGE
        end

        local actualTable = self:extractTable(tableName)

        if actualTable ~= nil then
            local printTimer = DevHelper.measureStart("Table printed in %.2f seconds")
            collectgarbage("collect") -- Ensure we have maximum mem available for this memory consuming operation
            Log:table(tableName, actualTable, maxDepth)
            collectgarbage("collect") -- Also cleanup to not leave a mess
            printTimer:stop()
        end
	end,

    visualizeTable = function (self, tableName, maxDepth)
        local USAGE = "USAGE: # dtVisualizeTable tableName [maxDepth]"
        if tableName == nil then
            Log:error("tableName is mandatory")
            return USAGE
        end
        maxDepth = tonumber(maxDepth)
        maxDepth = maxDepth or 2
        if type(maxDepth) ~= "number" then
            Log:error("maxDepth must be a number")
            return USAGE
        end

        local actualTable = self:extractTable(tableName)

        if actualTable ~= nil then
            local visualizeTimer = DevHelper.measureStart("Table visualized in %.2f seconds")
            collectgarbage("collect") -- Ensure we have maximum mem available for this memory consuming operation
            setFileLogPrefixTimestamp(false)
            print(tableName .. ":")
            DevHelper.visualizeTable(actualTable, maxDepth)
            collectgarbage("collect") -- Also cleanup to not leave a mess
            setFileLogPrefixTimestamp(g_logFilePrefixTimestamp)
            visualizeTimer:stop()

        end
    end,

    saveTable = function (self, tableName, filename, maxDepth, includeMods)
        local USAGE = "USAGE: # dtSaveTable tableName filename [maxDepth]"
        if tableName == nil then
            Log:error("tableName is mandatory")
            return USAGE
        end

        -- local maxDepth = 4
        maxDepth = tonumber(maxDepth)
        maxDepth = maxDepth or 2

        -- local includeModsOnGlobal = false
        includeMods = includeMods or false


        if not string.endsWith(filename, ".lua") then
            filename = filename .. ".lua"
        end

        local filePath = Utils.getFilenameFromPath(filename)

        if filePath == nil then
            filePath = filename
        end

        -- g_modSettingsDirectory
        filePath = l_currentModSettingsDirectory .. filePath


        local ignoredTables = {}
        -- local ignoredTableNames = {}

        -- Log:table("g_modManager.validMods", g_modManager.validMods)
        local actualTable = self:extractTable(tableName)

        local __G = DevTools.__g
        local isTrueGlobal = (actualTable == __G)

        -- Log:var("isTrueGlobal", isTrueGlobal)

        -- Omit mods from the global table
        if isTrueGlobal and not includeMods then
            -- Log:debug("Scanning %d valid mods", #g_modManager.validMods)
            for key, mod in ipairs(g_modManager.validMods) do
                -- Log:debug("%s", mod.modName)

                ignoredTables[mod.modName] = true
            end

            -- Log:debug("Scanning %d mods", #g_modManager.mods)
            -- Log:table("g_modManager.mods", g_modManager.mods)
            for key, mod in ipairs(g_modManager.mods) do
                -- Log:debug("%s", mod.modName)

                if ignoredTables[mod.modName] ~= true then
                    ignoredTables[mod.modName] = true
                else
                    Log:debug("Mod '%s' already added to the list", mod.modName)
                end
            end

            ignoredTables["g_modEventListeners"] = true
            ignoredTables["g_modIsLoaded"] = true
            ignoredTables["g_modNameToDirectory"] = true
        end

		createFolder(l_currentModSettingsDirectory)

        -- Log:debug("Saving '%s' to '%s' using maxDepth '%s' and allowing mods %s", tableName, filePath, maxDepth, tostring(includeMods))

        Log:info("Saving table '%s' [%s] to file '%s' using maxDepth %d", tableName, tostring(actualTable), filePath, maxDepth)

        if actualTable ~= nil then
            local saveTimer = DevHelper.measureStart("File saved in %.2f seconds")
            local memProfiler = MemoryProfiler.new()
            createFolder(l_currentModSettingsDirectory)

            DevTools:saveTable(tableName, actualTable, filePath, maxDepth, ignoredTables, memProfiler, OUTPUT_HEADER)

            local memUsed, memUsedMax = memProfiler:stop(true)
            saveTimer:stop(true)

            Log:info("Memory used: %.2f (peak %.2f) MB", memUsed / 1024, memUsedMax / 1024)

            if fileExists(filePath) then
                Log:info(saveTimer.results)
                -- Log:info("Saved '%s' to '%s' using maxDepth '%s' and allowing mods %s", tableName, filePath, maxDepth, tostring(includeMods))
            else
                Log:error("Failed to save '%s' to '%s'", tableName, filePath)
            end

        else
            Log:error("Table '%s' not found", tableName)
        end

	end,    
    testCommand = function (self)
        print("do nothing")
    end,
    clearLogAndConsole = function (self)
        self:clearLog()
        executeConsoleCommand("clear")
    end,
    clearLog = function (self)
        --TODO: add log rotator? maybe via config?
        local originalLogFile = getUserProfileAppPath() .. "log.txt"
        local tempLogFile = getUserProfileAppPath() .. "log.txt.tmp"
        local backupLogFile = getUserProfileAppPath() .. "log.bak"
        local function overwriteFile(filename, content)
            local fileId = DevTools.__g.createFile(filename, FileAccess.WRITE)
            fileWrite(fileId, content)
            delete(fileId)
            fileId = nil
        end
        local function doClearLog()
            copyFile(originalLogFile, backupLogFile, true)
            setFileLogName(tempLogFile) -- Temporarily redirect the log to a new file
            -- DevTools.__g.deleteFile(originalLogFile) -- Delete orignal log (via superglobal)
            overwriteFile(originalLogFile, "")
            collectgarbage("collect")
            setFileLogName(originalLogFile) -- Restore original log
            print("Log cleared")
        end

        local wasLogCleared = pcall(doClearLog)
        if not wasLogCleared then
            Log:warning("Failed to clear the log")
            setFileLogName(originalLogFile) -- Restore original log
        end
    end,
    exitGame = function (self)
        Log:debug("Exit game")
        doExit()
    end,
}

safeAddConsoleCommand("rr", "Restart the game", "hardReset", DevToolsConsoleCommands)
safeAddConsoleCommand("r", "Restart the game (soft restart)", "softRestart", DevToolsConsoleCommands)
safeAddConsoleCommand("tt", "Restart the game", "testRestart", DevToolsConsoleCommands)

safeAddConsoleCommand("dtRestartHard", "Restart the game", "hardReset", DevToolsConsoleCommands)
safeAddConsoleCommand("dtRestart", "Restart the game (soft restart)", "softRestart", DevToolsConsoleCommands)

g_powerTools.visualizeTable = function(self, tableName, tableObject, maxDepth)
    if tableObject == nil then
        Log:error("Table '%s' [%s] not found", tableName, tostring(tableObject))
        return
    end

    local visualizeTimer = DevHelper.measureStart("Table visualized in %.2f seconds")
    collectgarbage("collect") -- Ensure we have maximum mem available for this memory consuming operation
    setFileLogPrefixTimestamp(false)
    print(tableName .. ":")
    DevHelper.visualizeTable(tableObject, maxDepth)
    collectgarbage("collect") -- Also cleanup to not leave a mess
    setFileLogPrefixTimestamp(g_logFilePrefixTimestamp)
    visualizeTimer:stop()

end

g_powerTools.saveTable = function(self, tableName, actualTable, filePath, maxDepth, ignoredTables, header)
    header = header or OUTPUT_HEADER
    DevTools:saveTable(tableName, actualTable, filePath, maxDepth, ignoredTables, nil, header)
end


function DevTools:beforeLoadMap()
    safeAddConsoleCommand("rr", "Restart the game", "hardReset", DevToolsConsoleCommands)
    safeAddConsoleCommand("r", "Restart the game", "softRestart", DevToolsConsoleCommands)

    addConsoleCommand("dtTable", "Print table", "printTable", DevToolsConsoleCommands)
    addConsoleCommand("dtVisualizeTable", "Print table", "visualizeTable", DevToolsConsoleCommands)
    addConsoleCommand("dtPrintTable", "Print table to log", "printTable", DevToolsConsoleCommands)
    addConsoleCommand("dtSaveTable", "Save table to file", "saveTable", DevToolsConsoleCommands)

    addConsoleCommand("dtTest", "", "testCommand", DevToolsConsoleCommands)
    addConsoleCommand("dtClearLog", "", "clearLog", DevToolsConsoleCommands)

    xpcall(function()
        removeConsoleCommand("cls")
        addConsoleCommand("cls", "", "clearLogAndConsole", DevToolsConsoleCommands)
    end, function(err)
        
    end)
    

    addConsoleCommand("q", "", "exitGame", DevToolsConsoleCommands)
end

function DevTools:debugMultiplayerInfo()
    local debugInfo = {
        isMultiplayerGame = self:getIsMultiplayer(),
        isServer = self:getIsServer(),
        isClient = self:getIsClient(),
        isDedicatedServer = self:getIsDedicatedServer(),
        g_dedicatedServer = (g_dedicatedServer ~= nil),
        isMasterUser = self:getIsMasterUser(),
        hasFarmAdminAccess = self:getHasFarmAdminAccess(),
        isValidFarmManager = self:getIsValidFarmManager(),
    }

    DebugUtil.printTableRecursively(debugInfo, "Debub info:: ", 2)    
end

